/*=================================================================
 *
 * mgf_intAJD_c.c ...  Calculate the moment generating function of an integrated AJD. This function is vectorized
 *
 * The calling syntax is:
 *
 *		p = mgf_intAJD_c(x0, k, theta, sigma, L, mu, T, q)
 *
 * x0       ... initial value of AJD
 * k        ... mean reversion speed
 * theta    ... mean reversion level
 * L        ... jump intensity
 * mu       ... expected jump size of exponentially distributed jumps
 * T        ... time horizon
 * q        ... value at which to evaluate moment generating function (-1 for survival probabilities)
 *
 * sample: mgf_intAJD_c([0.01; 0.01], [0.1 0.25], [0.02; 0.02], [0.05; 0.05], [0.02; 0.02], [0.03; 0.03], [5; 5], [-1; -1])
 *
 *=================================================================*/

#include <math.h>
#include "mex.h"

#ifndef max
#define max(a,b) (((a) > (b)) ? (a) : (b))
#endif

/* Recursive computation of loss distribution */
static void mgf_intAJD_c(double x0[], double start[], double k[], double theta[], double sigma[], double L[], double mu[], double p[],
                         unsigned int num_dates, int T_max, double q[])
{
    int i, j, index;
    double gamma, c1, c2, d1, d2, A, B, T;
    double tmp, tmp2, tmp3, tmp4, tmp5, tmp6, tmp8;

    /* For each horizon and each data, calculate survival probability */
    for (i=0; i<num_dates; i++)
    {
        /* Check if q equal to zero */
        if ((q[i] > -1e-10) && (q[i] < 1e-10))
        {
            for (j=0; j<T_max; j++) 
            {
                p[i+j*num_dates] = 1;
            }
        } else {        
            /* For numerical stability, require sigma to be larger than 1e-5 */
            sigma[i] = max(sigma[i], 1e-5);

            /* Calculate a couple of auxiliary variables */
            gamma = sqrt(k[i]*k[i] - 2*sigma[i]*sigma[i]*q[i]);
            c1 = (gamma + k[i]) / (2*q[i]);
            c2 = 1 - mu[i]/c1;
            d1 = (-k[i] + gamma) / (2*q[i]); 
            d2 = (d1+mu[i])/c1;
            tmp = (-2*k[i]*theta[i]) / (sigma[i]*sigma[i]);
            tmp2 = L[i]*(c2*d1/c1-d2) / (-gamma*c2*d2);
            tmp3 = L[i]/c2-L[i];
            tmp4 = k[i]*theta[i]/c1;
            tmp5 = tmp3+tmp4;

            for (j=0; j<T_max; j++) 
            {
                T = start[i] + j*0.25;
                index = i+j*num_dates;
                tmp6 = exp(-gamma*T);
                tmp8 = (c1+d1*tmp6);
                A = tmp * log( tmp8/(c1 + d1) ) + tmp2 * log( (c2+d2*tmp6)/(c2+d2) ) + tmp5*T;
                B = (1-tmp6) / tmp8;
                p[index] = exp( A + x0[i] * B);
            }
        }
    }    
    return;
}


/* Gateway routine (to Matlab) */
void mexFunction( int nlhs, mxArray *plhs[], 
		  int nrhs, const mxArray*prhs[] )
     
{ 
    double *x0, *p, *horizons, *start; 
    double *k, *theta, *sigma, *L, *mu, *q;
    double tmp2;
    unsigned int num_dates, i; 
    int T_max;

    /* Get number of dates*/
    num_dates = mxGetM(prhs[0]); 
    
    /* Assign pointers to the input variables parameters */ 
    x0 = mxGetPr(prhs[0]);
    k = mxGetPr(prhs[1]);
    theta = mxGetPr(prhs[2]);
    sigma =mxGetPr(prhs[3]);
    L = mxGetPr(prhs[4]);
    mu = mxGetPr(prhs[5]);
    horizons = mxGetPr(prhs[6]);
    q = mxGetPr(prhs[7]);
    
    /* Determine maximum horizon in order to allocate right amount of memory */
    tmp2 = 0;
    for (i=0; i < num_dates; i++)
    {
        tmp2 = max(tmp2, horizons[i]);
    }
    T_max = (int) ceil(tmp2*4);
    
    /* Convert horizons to start_horizons, i.e. smallest positive number in years of the form horizon - k/4 */
    start = malloc(num_dates * sizeof(tmp2)); /* fast than: mxGetPr(mxCreateDoubleMatrix(num_dates, 1, mxREAL)); */
    for (i=0; i < num_dates; i++)
    {
        start[i] = horizons[i] - floor(horizons[i]*4)/4;
        if (start[i] == 0)
        {
            start[i] = 0.25;
        }
    }
    
    /* Create a mtrix for the return argument */ 
    plhs[0] = mxCreateDoubleMatrix(num_dates, T_max, mxREAL);
    p = mxGetPr(plhs[0]);
        
    /* Do the actual computations in a subroutine */
    mgf_intAJD_c(x0, start, k, theta, sigma, L, mu, p, num_dates, T_max, q);
    /* p[0] = q[0]; */
    /* p[1] = start[1]; */
    free(start);
    return;
}
